"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.MultitenancyLifecycle = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _url = require("url");
var _lodash = require("lodash");
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
var _multitenancy = require("../../../common/multitenancy");
/*
 *    Copyright 2020 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

class MultitenancyLifecycle {
  constructor({
    authManager,
    kerberos,
    searchGuardBackend,
    configService,
    sessionStorageFactory,
    logger,
    pluginDependencies,
    spacesService,
    kibanaCore,
    clusterClient
  }) {
    (0, _defineProperty2.default)(this, "onPreAuth", async (request, response, toolkit) => {
      const authType = this.configService.get('searchguard.auth.type');
      const debugEnabled = this.configService.get('searchguard.multitenancy.debug');
      const externalTenant = this.getExternalTenant(request, debugEnabled);

      // If we have an externalTenant, we will continue so that we can
      // update the cookie's tenant value
      if (!this.isRelevantPath(request) && !externalTenant) {
        return toolkit.next();
      }
      let selectedTenant = null;
      const {
        authHeaders,
        sessionCookie
      } = await this.getSession(request);
      const isAuthenticatedRequest = authType === 'proxy' || authHeaders && authHeaders.authorization ? true : false;

      // We may run into ugly issues with the capabilities endpoint here if
      // we let through an unauthenticated request, despite try/catch
      // Hence, only call the tenant endpoint if we are using proxy
      // or have an authorization header.
      if (!isAuthenticatedRequest) {
        return toolkit.next();
      }

      // Check if MT is enabled in the backend
      const {
        kibana_mt_enabled
      } = await this.searchGuardBackend.getKibanaInfoWithInternalUser();
      this.configService.set('searchguard.multitenancy.enabled', kibana_mt_enabled || false);

      // Skip early if MT is not enabled
      if (!kibana_mt_enabled) {
        return toolkit.next();
      }
      try {
        if (this.clusterClient.config.requestHeadersWhitelist.indexOf('sgtenant') === -1) {
          this.clusterClient.config.requestHeadersWhitelist.push('sgtenant');
        }
      } catch (error) {
        this.logger.error(`Multitenancy: Could not check headers whitelist ${request.url.pathname}. ${error}`);
      }

      // The capabilities route may break the entire screen if
      // we get a 401 when retrieving the tenants. So for the
      // default capabilities, we can just skip MT here.
      if (request.url.pathname.indexOf('capabilities') > -1 && request.url.searchParams.get('useDefaultCapabilities') === "true") {
        return toolkit.next();
      }
      let userTenantInfo;
      try {
        // We need the user's data from the backend to validate the selected tenant
        userTenantInfo = await this.searchGuardBackend.getUserTenantInfo(authHeaders);
        if (!userTenantInfo.data.multi_tenancy_enabled) {
          // MT is disabled, we don't need to do anything.
          // This should have been checked earlier though, so this is just a fail safe.
          return toolkit.next();
        }
        selectedTenant = await this.getSelectedTenant({
          request,
          sessionCookie,
          username: userTenantInfo.data.username,
          externalTenant,
          userTenantInfo
        });
      } catch (error) {
        this.logger.error(`Multitenancy: Could not get tenant info from ${request.url.pathname}. ${error}`);
      }
      const requestHasRequestedTenant = externalTenant || typeof sessionCookie.tenant !== 'undefined';
      // If we have an external tenant, but the selectedTenant is null
      // after validation, that means that the user does not have
      // access to the requested tenant, or it does not exist
      if (selectedTenant === null && requestHasRequestedTenant) {
        if (request.url.pathname.startsWith('/app') || request.url.pathname === '/') {
          // If we have the wrong tenant in the cookie, we need to reset the cookie tenant value
          const shouldResetCookieTenant = !externalTenant && typeof sessionCookie.tenant !== 'undefined';
          if (shouldResetCookieTenant) {
            delete sessionCookie.tenant;
            await this.sessionStorageFactory.asScoped(request).set(sessionCookie);
          }
          if (request.url.searchParams.get('sgtenantsmenu') !== _multitenancy.MISSING_TENANT_PARAMETER_VALUE) {
            return response.redirected({
              body: 'Wrong tenant',
              //statusCode: 401,
              headers: {
                'location': this.basePath + `/app/home?sgtenantsmenu=` + _multitenancy.MISSING_TENANT_PARAMETER_VALUE
              }
            });
          }
        } else {
          // TODO maybe just pass through and let the backend return an error?
          return response.unauthorized();
        }
      }

      // We have a tenant to use
      if (selectedTenant !== null) {
        const rawRequest = (0, _coreHttpRouterServerInternal.ensureRawRequest)(request);
        (0, _lodash.assign)(rawRequest.headers, authHeaders, {
          sgtenant: selectedTenant
        });
        if (this.pluginDependencies.spaces) {
          // If we have a new tenant with no default space, we need to create it.
          await this.spacesService.createDefaultSpace({
            request,
            selectedTenant
          });
        }
        if (selectedTenant !== sessionCookie.tenant) {
          // save validated tenant in the cookie
          sessionCookie.tenant = selectedTenant;
          await this.sessionStorageFactory.asScoped(request).set(sessionCookie);
        }
      } else {
        let authRequiredForRoute = false;
        try {
          authRequiredForRoute = request.route.options.authRequired;
        } catch (error) {
          // Ignore
        }

        // Could also be "optional" or false
        if (authRequiredForRoute === true) {
          return response.redirected({
            body: 'Missing Tenant',
            //statusCode: 401,
            headers: {
              'location': this.basePath + `/searchguard/login?err=` + _multitenancy.MISSING_TENANT_PARAMETER_VALUE
            }
          });
        }
      }
      return toolkit.next();
    });
    /**
     * Get and validate the selected tenant.
     * @param request
     * @param sessionCookie
     * @param {string} username
     * @param {string|null} externalTenant
     * @param userTenantInfo
     * @returns {Promise<string|null>}
     */
    (0, _defineProperty2.default)(this, "getSelectedTenant", async ({
      request,
      sessionCookie,
      username,
      externalTenant,
      userTenantInfo
    }) => {
      const debugEnabled = this.configService.get('searchguard.multitenancy.debug');
      const backend = this.searchGuardBackend;

      // default is the tenant stored in the tenants cookie
      let selectedTenant = sessionCookie && typeof sessionCookie.tenant !== 'undefined' ? sessionCookie.tenant : null;
      if (debugEnabled) {
        this.logger.info(`Multitenancy: tenant_storagecookie ${selectedTenant}`);
      }
      if (externalTenant) {
        selectedTenant = externalTenant;
      }

      /**
       * @type {Record<string, boolean>}
       */
      const userTenants = this.searchGuardBackend.convertUserTenantsToRecord(userTenantInfo.data.tenants);

      // if we have a tenant, check validity and set it
      if (typeof selectedTenant !== 'undefined' && selectedTenant !== null && selectedTenant !== "") {
        selectedTenant = backend.validateRequestedTenant(username, selectedTenant, userTenants);
      } else if (userTenantInfo && userTenantInfo.data.default_tenant) {
        selectedTenant = userTenantInfo.data.default_tenant;
      }
      if (debugEnabled) {
        this.logger.info(`Multitenancy: tenant_assigned ${selectedTenant}`);
      }
      return selectedTenant;
    });
    /**
     * Get the auth information needed to make user scoped requests
     * @param request
     * @returns {Promise<{sessionCookie: {}, authHeaders: *}>}
     */
    (0, _defineProperty2.default)(this, "getSession", async request => {
      let sessionCookie;
      let authHeaders = request.headers;
      if (this.authManager) {
        const authInstance = await this.authManager.getAuthInstanceByRequest({
          request
        });
        if (authInstance) {
          sessionCookie = await authInstance.getCookieWithCredentials(request);
          authHeaders = authInstance.getAuthHeader(sessionCookie);
        } else {
          sessionCookie = await this.sessionStorageFactory.asScoped(request).get();
        }
      } else if (this.kerberos) {
        sessionCookie = await this.kerberos.getCookieWithCredentials(request);
        authHeaders = this.kerberos.getAuthHeader(sessionCookie);
      } else {
        sessionCookie = await this.sessionStorageFactory.asScoped(request).get();
      }
      if (!sessionCookie) {
        sessionCookie = {};
      }
      return {
        sessionCookie,
        authHeaders
      };
    });
    (0, _defineProperty2.default)(this, "isRelevantPath", request => {
      const path = request.url.pathname;

      // MT is only relevant for these paths
      const relevantPaths = ['/internal', '/goto', '/opensearch', '/app', '/api', '/bootstrap.js'];

      // MT is not relevant in these patterns
      const ignorePatterns = ['/api/status', '/api/v1/auth/config', '/api/v1/auth/login', '/api/v1/systeminfo'];
      return path === '/' || relevantPaths.some(root => path.startsWith(root)) && !ignorePatterns.some(root => path.startsWith(root));
    });
    /**
     * Check if we have a tenant set as query parameter
     * or as header value
     *
     * @param request
     * @param debugEnabled
     * @returns {null|string}
     */
    (0, _defineProperty2.default)(this, "getExternalTenant", (request, debugEnabled = false) => {
      let externalTenant = null;
      // check for tenant information in HTTP header. E.g. when using the saved objects API
      if (request.headers.sgtenant || request.headers.sg_tenant) {
        externalTenant = request.headers.sgtenant ? request.headers.sgtenant : request.headers.sg_tenant;
        if (debugEnabled) {
          this.logger.info(`Multitenancy: tenant_http_header: ${externalTenant}`);
        }
      }
      // check for tenant information in GET parameter. E.g. when using a share link. Overwrites the HTTP header.
      if (request.url.searchParams.has('sg_tenant') || request.url.searchParams.has('sgtenant')) {
        externalTenant = request.url.searchParams.has('sg_tenant') ? request.url.searchParams.get('sg_tenant') : request.url.searchParams.get('sgtenant');
        if (debugEnabled) {
          this.logger.info(`Multitenancy: tenant_url_param' ${externalTenant}`);
        }
      }
      if (externalTenant !== null) {
        try {
          if (externalTenant.toLowerCase() === 'private') {
            return _multitenancy.PRIVATE_TENANT_NAME;
          }
          if (externalTenant.toLowerCase() === 'global' || externalTenant.toUpperCase() === _multitenancy.GLOBAL_TENANT_NAME) {
            return _multitenancy.GLOBAL_TENANT_NAME;
          }
        } catch (error) {
          this.logger.error(`Could not translate private/global tenant: ` + externalTenant);
        }
      }
      return externalTenant;
    });
    this.authManager = authManager;
    this.searchGuardBackend = searchGuardBackend;
    this.configService = configService;
    this.sessionStorageFactory = sessionStorageFactory;
    this.logger = logger;
    this.pluginDependencies = pluginDependencies;
    this.spacesService = spacesService;
    this.kerberos = kerberos;
    this.kibanaCore = kibanaCore;
    this.clusterClient = clusterClient;
    this.basePath = kibanaCore.http.basePath.get();
  }
}
exports.MultitenancyLifecycle = MultitenancyLifecycle;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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